/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2006 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- CVS Information ---
 *  $Id: Environment.java 3362 2006-03-30 22:27:26Z gregork $
 */
package phex.common;

import java.io.File;
import java.io.IOException;
import java.util.*;

import org.apache.commons.lang.SystemUtils;

import phex.event.UserMessageListener;
import phex.utils.NLogger;
import phex.utils.NLoggerNames;
import phex.utils.VersionUtils;

/**
 * This class can not be implemented as a manager since manager initialization
 * routines relay on the availability of this class during initialization.
 */
public class Environment
{
    private Properties properties;
    private File configurationRoot;
    
    private UserMessageListener userMessageListener;

    /**
     * The TimerService is a single thread that will handle multiple TimerTask.
     * Therefore each task has to make sure it is not performing a long blocking
     * operation.
     */
    private Timer timerService;
    private String javaVersion;

    private Environment()
    {
        try
        {
            properties = new Properties();
            properties.load( Environment.class.getResourceAsStream(
                "/phex/resources/version.properties" ) );
        }
        catch ( IOException exp )
        {
            NLogger.error( NLoggerNames.STARTUP, exp, exp );
            throw new RuntimeException();
        }
        initializeJavaVersion();
        timerService = new Timer( true );
    }

    static private class Holder
    {
        static protected final Environment environment = new Environment();
    }

    static public Environment getInstance()
    {
        return Environment.Holder.environment;
    }

    /**
     * Sets the directory into which Phex adds its configuration files. When
     * configRoot is null the directory is set to:<br>
     * {user.home}/phex on windows systems and<br>
     * {user.home}/.phex on unix and mac systems.
     * @param configRoot the directory into which Phex adds its configuration files
     *        or null.
     */
    public void setPhexConfigRoot( File configRoot )
        throws IOException
    {
        if ( configRoot == null )
        {
            StringBuffer path = new StringBuffer(20);
            path.append( System.getProperty("user.home") );
            path.append( File.separator );

            //phex config files are hidden on all UNIX systems (also MacOSX. Since
            //there are many UNIX like operation systems with Java support out there,
            //we can not recognize the OS through it's name. Thus we check if the
            //root of the filesystem starts with "/" since only UNIX uses such
            //filesystem conventions
            if ( File.separatorChar == '/' )
            {
                path.append ('.');
            }
            path.append ("phex");
            configRoot = new File( path.toString() );
        }
        if ( !configRoot.isDirectory() )
        {
            boolean succ = configRoot.mkdirs();
            if ( !succ )
            {
                throw new IOException( "Cant create directory: " + configRoot.getAbsolutePath() );
            }
        }
        configurationRoot = configRoot;
    }

    /**
     * Returns the File representing the complete path to the configuration file
     * with the given configFileName.
     * @param configFileName the name of the config file to determine the complete
     *        path for.
     * @return the File representing the complete path to the configuration file
     *         with the given configFileName.
     */
    public File getPhexConfigFile( String configFileName )
    {
        if ( configurationRoot == null )
        {
            try
            {
                setPhexConfigRoot( null );
            }
            catch ( IOException exp )
            {// TODO test if logging here can cause trouble when the Logger cant
             // initialize the file name.
             //Logger.logError( exp );
            }
        }
        return new File( configurationRoot, configFileName );
    }

    public String getProperty( String name )
    {
        return properties.getProperty( name );
    }

    public String getProperty( String name, String defaultValue )
    {
        return properties.getProperty( name, defaultValue );
    }

    /**
     * Returns the Phex Vendor string containing the Phex version.
     * @return the Phex Vendor string containing the Phex version.
     */
    public static String getPhexVendor()
    {
        return "Phex " + Cfg.PRIVATE_BUILD_ID + VersionUtils.getFullProgramVersion();
    }

    /**
     * Schedules the specified task for repeated fixed-delay execution,
     * beginning after the specified delay. Subsequent executions take place at
     * approximately regular intervals separated by the specified period.
     *
     * The TimerService is a single thread that will handle multiple TimerTask.
     * Therefore each task has to make sure it is not performing a long blocking
     * operation.
     *
     * @param task The task to be scheduled.
     * @param delay The delay in milliseconds before task is to be executed.
     * @param period The time in milliseconds between successive task executions.
     */
    public void scheduleTimerTask(TimerTask task, long delay, long period )
    {
        timerService.schedule( task, delay, period );
    }
    
    /**
     * Schedules the specified task for execution after the specified delay.
     * 
     * The TimerService is a single thread that will handle multiple TimerTask.
     * Therefore each task has to make sure it is not performing a long blocking
     * operation.
     *
     * @param task The task to be scheduled.
     * @param delay The delay in milliseconds before task is to be executed.
     */
    public void scheduleTimerTask(TimerTask task, long delay )
    {
        timerService.schedule( task, delay );
    }

    /**
     * Returns true if the system is a ultrapeer os, false otherwise.
     * @return true if the system is a ultrapeer os, false otherwise.
     */
    public boolean isUltrapeerOS()
    {
        // accept all none windows systems (MacOSX, Unix...) or Windows 2000 or XP.
        return !SystemUtils.IS_OS_WINDOWS || 
            SystemUtils.IS_OS_WINDOWS_2000 || SystemUtils.IS_OS_WINDOWS_XP;
    }
    
    private void initializeJavaVersion()
    {
        javaVersion = System.getProperty( "java.version" );
    }
        
    /**
     * Checks if the user runs a Java 1.4 or later VM.
     */
    public boolean isJava14orLater()
    {
        return !javaVersion.startsWith("1.3") && !javaVersion.startsWith("1.2") 
            && !javaVersion.startsWith("1.1") && !javaVersion.startsWith("1.0"); 
    }
    
    public void setUserMessageListener( UserMessageListener listener )
    {
        userMessageListener = listener;
    }
    
    public void fireDisplayUserMessage( String userMessageId )
    {
        // if initialized
        if ( userMessageListener != null )
        {
            userMessageListener.displayUserMessage( userMessageId, null );
        }
    }
    
    public void fireDisplayUserMessage( String userMessageId, String[] args )
    {
        // if initialized
        if ( userMessageListener != null )
        {
            userMessageListener.displayUserMessage( userMessageId, args );
        }
    }
}