#/usr/bin/env python
# encoding: utf-8

"""Simulate a swarm, object driven. 

Every swarm member is an object/an instance of a class. 

The Scene is an object, too. 

For base usage, please look at the Self-Test at the bottom of the page. 

You can find the main Parameters inside the SwarmBlob class as attributes. 

Ideas: 
    - Male and female blobs (blue and green?). 
    - Create new blobs just before fleeing, if your partner was of the other sex. Die after 5 broken hearts. 
    - Make them flee once their images touch, not when they get too close. 
"""

# First we need PyGlet,  window and image for the dot

from pyglet import window,  image

# For the alpha Channel we also need pyglet.gl

from pyglet.gl import *

# And we need the clock for fixed frames per second. 

from pyglet.clock import tick as clock_tick

# Then we need a random movement, which means here: randint

from random import randint

# And while we're at it, we also get the choice functino, to choose a random blob from a list. 

from random import choice

# For loading files system independently, we also need the path methods from os. 

from os import path

#### Constants ####

IMAGE_BACKGROUND = "blob_heart_bg.png" #: The Background image. 


# As preparation we create a scripte-wide list to hold all blobs. 

blobs = [] #: This list contains all blobs. 

# Now we first create the scene class as a general class to write easy games. 

class Scene(window.Window): 
    """The base class for any game. 
    
    Init the screen and an event loop which evokes the function self.action() for any action of the game."""
    def __init__(self,  scene_name="Default Scene",  width=640,  height=480,  fullscreen=False,  resizable=False, image_base_folder="images",   background=IMAGE_BACKGROUND,  fps=60): 
        if not fullscreen: 
            super(Scene,  self).__init__(width,  height,  fullscreen=False,  resizable=resizable)
        else: 
            super(Scene,  self).__init__(fullscreen=fullscreen)
        self.scene_name = str(scene_name)
        # First we enable OpenGL for alpha channel
        glEnable(GL_BLEND)
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)
        # Add a background, if one is given. 
        self.background = background #: The background of the window. 
        if self.background is not None: 
            self.background = image.load(self.media_path(image_base_folder,  background))
        self.frames_per_second = fps #: The frames per second we want
        self.frame_display_time = 1.0 / self.frames_per_second #: The time a frame is being displayed. 
        self.dt = 0 #: The time passed (delta t)
        # Then we enter the event_loop. 
        # self.event_loop() # this should be done in the subclass, so we can access attributes of the Scene and Window class (else any new attribute must be specified before calling the __init__ of teh Scene class). 
    
    def media_path(self,  base_folder,  source):
        """Get the full path from base_folder and source name."""
        return path.join(path.join(path.dirname(__file__),  base_folder),  source)
        
    # Now we need an action: 
    def action(self):
        """Do what the script should do in the event loop."""
        raise NotImplementetException("An action in the event_loop must be implemented for a scene to do anything.")
        
    # Later on, we'll also need the event loop. 
    def event_loop(self):
        while not self.has_exit: 
            self.dispatch_events()
            self.dt += clock_tick()
            if self.dt >= self.frame_display_time: 
                self.clear()
                if self.background is not None: 
                    self.background.blit(0, 0, -1)
                self.action()
                self.flip()
                self.dt = 0
        

# We create the BlobSwarm as a Scene

class BlobSwarm(Scene):
    """The Scene of the Blob Swarm. 
    
    Gathers the blobs and supplies the action() function which makes the blobs move around. """
    def __init__(self,  mode="dont't get too close", width=640,  height=480,  fullscreen=False):
        self.mode = mode
        super(BlobSwarm,  self).__init__(scene_name="Blob Swarm",  width=width,  height=height,  fullscreen=fullscreen)
        # We prepared everything, so we enter the event loop. 
        self.event_loop()

        
    def action(self):
        """Make the blobs move around. 
        
        Let them choose their new position and then repaint them."""
        for i in blobs: 
            i.move_around(width=self.width,  height=self.height,  mode=self.mode)
            i.image.blit(i.pos_x,  i.pos_y)
#        for i in shadowblobs: 
#            i.image.blit(i.pos_x,  i.pos_y)

# Now we create a SwarmBlob class, which supplies all the necessary functions for the blobs actions: At the moment "Move" 

class SwarmBlob(object):
    """A single Blob in a swarm. 
    
    It can choose its next position and knows the other blobs.
    
    Details about the movement are written in the method move_around()
    """
    def __init__(self,  x=320,  y=240,  safe_range=8,  search_range=16,  flee_steps=20):
        # SwarmBlobs must have an x and an y position
        # They all begin at the same spot. 
        # Idea: Begin at a randomized spot: x + randint(0,  10) - 5
        self.pos_x = x #: The position of the blob on the x-axis
        self.pos_y = y #: The position of the blob on the y-axis
        # Also they need an image as visible representation
        self.image_base_folder = path.join(path.dirname(__file__),  "images")
        self.image = image.load(path.join(self.image_base_folder,  "blob_heart.png"))
        self.safe_range = safe_range #: The safe distance. Blobs will jump away if others get closer than this. 
        self.search_range = search_range #: The range in which the blob will first serch for partners. 
        self.steps_to_move = 0 #: Steps the blob still has to move instead of doing other things. 
        self.flee_steps = flee_steps #: The number of steps to run when fleeing from a partner. 
        self.partner = None #: An optional partner for the blob. 
        self.former_partner = None #: An optional attribut to store the former partner, for example to avoid connecting to the same partner again at once. 
    
    def squared_distance(self,  other):
        """Return the squared distance between this blob and another one.
        
        This should be significantly faster than returning the distance, which is important since this function gets calles very often. """
        return (self.pos_x - other.pos_x)**2 + (self.pos_y - other.pos_y)**2
        
    # also they need a movement method. 
    def move_around(self, width=640,  height=480,  mode="don't get too close"):
        """Move around in different modes.
        
        This function serves as a switch between several movement modes.
        
        the current single mode is "don't get to close". 
        
        If self.steps_to_move is higher than 0, it moves the blob one random step instead of starting other action. """
        
        if self.steps_to_move != 0: 
            self.random_move(width,  height)
            self.steps_to_move -= 1
        elif mode == "don't get too close": 
            self.movement_not_too_close(blobs,  width,  height)
        elif mode == "blah": 
            self.move_blah()
        elif mode == "valentine": 
            self.move_shadow_valentine(width,  height)
    
    def move_blah(self,  blobbs=blobs):
        """Blah movement"""
        # Copy the list of blobs to be able to remove myself without removing me from the global list
        blabs = blobbs[:]
        # Remove myself from the list of blobs
        if self in blabs: 
            blabs.remove(self)
        # Now move closer to each blob
        for i in blabs: 
            # calculate the distance in x direction between myself and the blob
            x_dist = (i.pos_x - self.pos_x )
            # Move 1/300th of the distance towards the other. 
            self.pos_x += x_dist / 30.0
            # calculate the distance in y direction between myself and the blob
            y_dist = (i.pos_y - self.pos_y )
            # Move 1/300th of the distance towards the other. 
            self.pos_y +=  y_dist / 30.0
            # If the blob is too close, move one step at random next round. 
            if x_dist < self.safe_range: 
                self.steps_to_move += 1
            if y_dist < self.safe_range: 
                self.steps_to_move += 1
            
    def move_shadow_valentine(self,  width,  height): 
        """Move close to the shadow blobs"""
        #self.move_blah(blobbs=[choice(blobs), choice(shadowblobs)])
        self.movement_not_too_close(blobs+shadowblobs,  width,  height)
    
    def random_move(self,  width,  height):
        """Move at random on the screen.
        
        At a screen width/height of 640/480, we move up to 12.8/9.6 pixels away at an average of 6.4/4.8 pixels per step. """
        self.pos_x += randint(0,  width) / 25.0 - width / 50.0
        self.pos_y += randint(0,  height) / 25.0 - height / 50.0
    
    def break_partnership(self):
        """Break any existing partnership"""
        if self.partner is not None: 
            # set current parnter as former partner
            self.former_partner = self.partner
            # remove current partner, so you can take another partner. 
            self.partner = None
        # Remove yourself from being partner of your former partner. 
        if self.former_partner is not None and self.former_partner.partner is not None: 
            # Set yourself as former partner of your former partner. 
            self.former_partner.former_partner = self
            # remove yourself from being partner of your former_partner. 
            self.former_partner.partner = None
            
        
    def choose_partner(self,  blobs):
        """Choose a partner 
        
        Choose from all those blobs within your search distance, who don't yet (no more) have a partner and aren't your former partner. 
        
        If you find one, half your search distance. 
        
        If no partner is avaible, square your search distance (see comments for safeguards) and make sure, you are set as free (self.partner = None). """
        # Search for a partner in your search distance. 
        close_avaible_blobs = []
        for i in blobs: 
            # is any fitting partner is avaible, add it. 
            if self.squared_distance(i) <= self.search_range**2 and i.partner is None and self.former_partner is not i: 
                close_avaible_blobs.append(i)
        # If there are avaible partners, choose one of them. 
        if len(close_avaible_blobs) > 0: 
            self.partner = choice(close_avaible_blobs)
            self.partner.partner = self
            # make search range smaller, since you found one in your range
            self.search_range = self.search_range / 2.0
        else: 
            self.partner = None
            # make sure the search range is higher than 1
            if self.search_range < 1: 
                self.search_range = 1.0 / self.search_range 
            if self.search_range == 1: 
                self.search_range = 1.1
            if self.search_range > 2400: # Maximum sensible width of a window I can think of.  This is necessary to avoid the error: 
                # OverflowError: (34, 'Numerical result out of range') 
                # in _choose_partner where search_range gets squred. 
                self.search_range = 2400
            # square search range
            self.search_range = self.search_range**2
        
    
    def movement_not_too_close(self,  blobs,  width,  height):
            """All try to gather, but are restricted by not letting their respective partner into their safe distance.
            
            As soon as the partner enters the safe_distance, the blob breaks the partnership, runs away for some time and then seeks a new partner. """
            # first choose a partner, if you don't already have one, if you're your own partner or your partner doesn't recognize you as partner anymore. 
            if self.partner is None or self.partner is self or self.partner.partner is not self: 
                # Select a partner
                self.choose_partner(blobs)
            # If another blob is in the safe_range: Jump a random step away, up to 10% of the screen, the fast part. Simple a**2 + b**2 = c**2
            if self.partner is not None and self.squared_distance(self.partner) < self.safe_range ** 2 and self is not i: 
                self.steps_to_move = self.flee_steps # The blob will move a number of random steps equal to its flee_steps before participating again. 
                # And since the partner got to close, choose another one. 
                self.break_partnership()
                self.choose_partner(blobs)
            # Else, if you have a partner
            elif self.partner is not None: 
                # else move a a bit towards it, the slow part. 
                self.pos_x += (self.partner.pos_x - self.pos_x) / 64.0
                self.pos_y += (self.partner.pos_y - self.pos_y) / 64.0
            # In any case, move around a bit just to stay moving. This drains too much resources and isn't really needed. 
            self.pos_x += (randint(0,  2) -1 ) / 10.0
            self.pos_y += (randint(0,  2) -1 ) / 10.0
            
            self.keep_on_screen(width,  height)
    
    def keep_on_screen(self,  width=640,  height=480):
        """If the blob reaches the edge of the screen, move it back to the edge (the dynamics should keep them away from the edges automatically then). """
        # If it is too far to the right, 
        if self.pos_x > width: 
            # set it at the edge. 
            self.pos_x = width
        # If it is too far to the left, 
        elif self.pos_x < 0: 
            # set it to the edge
            self.pos_x = 0
        # same with top and bottom. 
        if self.pos_y > height: 
            self.pos_y = height
        elif self.pos_y < 0: 
            self.pos_y = 0

def create_valentine_blobs():
    """Create a number of shadowblobs (not shown blobs) which form a heart."""
    shadowblobs = []
    # Innere Spitze
    shadowblobs.append(SwarmBlob(x=640,  y=648))
    # Rundung links
    shadowblobs.append(SwarmBlob(x=460,  y=760))
    # Rundung rechts
    shadowblobs.append(SwarmBlob(x=820,  y=760))
    # links oben
    shadowblobs.append(SwarmBlob(x=320,  y=648))
    # oben rechts
    shadowblobs.append(SwarmBlob(x=960,  y=648))
    # Mitte unten
    shadowblobs.append(SwarmBlob(x=640,  y=200))
    # drei zwischenblobs links
    shadowblobs.append(SwarmBlob(x=400,  y=536))
    shadowblobs.append(SwarmBlob(x=480,  y=424))
    shadowblobs.append(SwarmBlob(x=560,  y=312))
    # drei zwischenblobs rechts
    shadowblobs.append(SwarmBlob(x=880,  y=536))
    shadowblobs.append(SwarmBlob(x=800,  y=424))
    shadowblobs.append(SwarmBlob(x=720,  y=312))
    return shadowblobs

#### Self-Test ####
        
# if this file is called directly, we execute teh valentine scene with 19 blobs. 

if __name__ == "__main__": 
    # First we create the heart shape of blobs which won't be shown. 
    shadowblobs = create_valentine_blobs()
    # Then we create all other blobs
    for i in range(19): 
        # We create the blob at a random position. 
        blob = SwarmBlob(x=randint(0,  1280),  y=randint(0,  960))
        # And add it to the script wide list of blobs. 
        blobs.append(blob)
    # At the end we create the scene which will show the blobs. 
    # It is started in valentine mode => Displaying a heart for my heart. 
    scene = BlobSwarm(mode="valentine",  width=1280,  height=960,  fullscreen=True)
    
    # For my love
    print "Für Verónica"
    
