#
# Copyright (C) 2007 Johan Dahlin
#
# All rights reserved
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
# USA.
#

import atexit
import errno
import os
import signal
import socket

from launcher.common import DAEMON_SOCKET
from launcher.support import get_supported_modules

class Daemon(object):
    def __init__(self):
        self.pid = os.getpid()

        modules = get_supported_modules()
        for module in modules:
            module.daemon_init()
        self._modules = modules

        signal.signal(signal.SIGPIPE, signal.SIG_IGN)
        signal.signal(signal.SIGCHLD, self._signal_sigchild)

    def _signal_sigchild(self, signum, frame):
        while True:
            try:
                childpid, status = os.waitpid(-1, os.WNOHANG)
            except OSError, e:
                # No child processes
                if e.errno == errno.ECHILD:
                    break

            if childpid == 0:
                break

    def _read_request(self, conn):
        fd = conn.makefile()
        program = fd.readline()[:-1]
        data = fd.readline()[:-1]
        args = []
        while data != "":
            args.append(data)
            data = fd.readline()[:-1]

        self._fork(conn, program, args)

    def _fork(self, conn, program, args):
        pid = os.fork()
        if pid:
            return

        signal.signal(signal.SIGPIPE, signal.SIG_DFL)
        signal.signal(signal.SIGCHLD, signal.SIG_DFL)

        for module in self._modules:
            module.client_init(program)

        glob = dict(__name__="__main__")
        print 'launching', program
        execfile(program, glob, glob)

        raise SystemExit

    def _daemon_exit(self):
        if os.getpid() != self.pid:
            return

        if os.path.exists(DAEMON_SOCKET):
            os.unlink(DAEMON_SOCKET)

    def run(self):
        if os.path.exists(DAEMON_SOCKET):
            os.unlink(DAEMON_SOCKET)

        sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        sock.bind(DAEMON_SOCKET)
        atexit.register(self._daemon_exit)

        sock.listen(1)

        while True:
            try:
                conn, addr = sock.accept()
            except socket.error, e:
                # Interrupted system call
                if e[0] == errno.EINTR:
                    continue

            self._read_request(conn)
            conn.close()
